import cloudinary from '../helpers/cloudinary.js';
import fs from 'fs';
import path from 'path';
import Teacher from '../models/teacherModel.js';

// create Teacher
export const createTeacher = async (req, res, next) => {
  try {
    const {
      teacher_name,
      teacher_name_bd,
      designation,
      nid,
      index,
      phone_number,
      address,
      joining_date,
      retirement_date,
      date_of_birth,
      gender,
    } = req.body;
    console.log(req.body);

    // Validation
    if (
      !teacher_name ||
      !designation ||
      !teacher_name_bd ||
      !nid ||
      !joining_date ||
      !retirement_date ||
      !index ||
      !phone_number ||
      !address ||
      !date_of_birth ||
      !gender
    ) {
      return res.status(500).send({ message: 'All fields are required' });
    }

    // Images File Handling
    const urls = [];
    const file = req.file;

    if (file) {
      const { path } = file;
      const result = await cloudinary.uploader.upload(path);
      const data = {
        url: result.secure_url,
        public_id: result.public_id,
      };
      urls.push(data);
      fs.unlinkSync(path);
    }

    const teacher = await Teacher.create({
      teacher_name,
      teacher_name_bd,
      designation,
      nid,
      index,
      phone_number,
      address,
      joining_date,
      retirement_date,
      date_of_birth,
      gender,
      image: urls,
    });

    res.status(201).send({
      success: true,
      message: 'Teacher created successfully',
      teacher,
    });
  } catch (error) {
    res.status(404).send({
      success: false,
      message: 'Something wrong!! Try again',
    });
    next(error);
    console.error(error);
  }
};

//get all Teacher
export const getTeacher = async (req, res) => {
  try {
    const teachers = await Teacher.find({});

    res.status(200).send({
      success: true,
      message: 'All Teacher get successfully ',
      teachers,
    });
  } catch (error) {
    console.log(error);
  }
};
//get single Teacher
export const getSingleTeacher = async (req, res) => {
  try {
    const id = req.params.id;
    const teacher = await Teacher.findById({ _id: id });
    res.status(200).send({
      success: true,
      message: 'Single Teacher get successfully ',
      teacher,
    });
  } catch (error) {
    console.log(error);
  }
};

//delete Teacher
export const deleteTeacher = async (req, res) => {
  try {
    const id = req.params.id;
    const teacher = await Teacher.findById({ _id: id });
    if (teacher.image) {
      for (const image of teacher.image) {
        const { public_id } = image;
        await cloudinary.uploader.destroy(public_id);
      }
    }
    await Teacher.findByIdAndDelete(id);

    res.status(200).send({
      success: true,
      message: 'Teacher deleted successfully',
    });
  } catch (error) {
    console.log(error);
    res.status(500).send({
      success: false,
      message: 'Error while deleting Student',
      error,
    });
  }
};
//update teacher
export const updateTeacher = async (req, res) => {
  try {
    const {
      teacher_name,
      teacher_name_bd,
      nid,
      joining_date,
      retirement_date,
      designation,
      index,
      phone_number,
      address,
      date_of_birth,
      gender,
    } = req.body;

    // console.log("",req.body);

    const id = req.params.id;

    // console.log('id========', id);
    // Validation
    if (
      !teacher_name ||
      !teacher_name_bd ||
      !nid ||
      !designation ||
      !retirement_date ||
      !joining_date ||
      !index ||
      !phone_number ||
      !address ||
      !date_of_birth ||
      !gender
    ) {
      return res.status(500).send({ message: 'All fields are required' });
    }

    const oldTeacher = await Teacher.findById({ _id: id });
    // console.log('old student ===========', oldStudent);
    // image handling
    const file = req.file;
    // console.log('file student ===========', file);

    const urls = file !== undefined ? [] : oldTeacher.image;

    if (file) {
      // delete old images
      for (const image of oldTeacher.image) {
        const { public_id } = image;
        await cloudinary.uploader.destroy(public_id);
      }

      const { path } = file;
      const result = await cloudinary.uploader.upload(path);
      const data = {
        url: result.secure_url,
        public_id: result.public_id,
      };
      urls.push(data);
      fs.unlinkSync(path);
    }

    const teacher = await Teacher.findByIdAndUpdate(
      id,
      {
        teacher_name,
        teacher_name_bd,
        nid,
        joining_date,
        retirement_date,
        designation,
        index,
        phone_number,
        address,
        date_of_birth,
        gender,
        qualification: oldTeacher.qualification,
        image: urls,
      },
      { new: true }
    );

    await teacher.save();

    res.status(201).send({
      success: true,
      message: 'teacher Updated Successfully',
      teacher,
    });
  } catch (error) {
    res.status(404).send({
      success: false,
      message: 'Something wrong!! Try again',
      error: error,
    });
    console.error(error);
  }
};

// ===================qualification Teacher======================

// qualification
export const pushQualification = async (req, res) => {
  try {
    const { name_exam, session, result, gpa, institute, board, group } =
      req.body;
    const teacher = await Teacher.findById(req.params.id);

    teacher.qualification.push({
      name_exam,
      session,
      result,
      gpa,
      institute,
      board,
      group,
    });
    await teacher.save();

    res.status(201).send({
      success: true,
      message: 'Qualification is added',
      teacher,
    });
  } catch (error) {
    console.log(error);
  }
};
// update qualification
export const updateQualification = async (req, res) => {
  try {
    const {
      tid,
      qid,
      name_exam,
      session,
      result,
      gpa,
      institute,
      board,
      group,
    } = req.body;

    await Teacher.updateOne(
      { _id: tid, 'qualification._id': qid },
      {
        $set: {
          'qualification.$.name_exam': name_exam,
          'qualification.$.session': session,
          'qualification.$.result': result,
          'qualification.$.gpa': gpa,
          'qualification.$.institute': institute,
          'qualification.$.board': board,
          'qualification.$.group': group,
        },
      }
    );

    res.status(200).send({
      success: true,
      message: 'Qualification is updated',
    });
  } catch (error) {
    console.log(error);
    res.status(500).send({
      success: false,
      message: 'Server error',
    });
  }
};

export const deleteQualification = async (req, res) => {
  try {
    const { id, qid } = req.body;

    await Teacher.updateOne(
      { _id: id },
      { $pull: { qualification: { _id: qid } } }
    );
    res.status(202).send({
      success: true,
      message: 'Qualification is deleted',
    });
  } catch (error) {
    console.log(error);
  }
};
