import cloudinary from '../helpers/cloudinary.js';
import fs from 'fs';
import path from 'path';
import Student from '../models/studentModel.js';

// create Student
export const createStudent = async (req, res, next) => {
  try {
    const {
      student_name,
      student_name_bd,
      father_name,
      mother_name,
      father_name_nid,
      mother_name_nid,
      uniq_id,
      sub_scholar_id,
      scholar_id,
      class_name,
      class_roll,
      phone_number,
      address,
      date_of_birth,
      birth_reg_no,
      gender,
      blood_group,
    } = req.body;

    // Validation
    if (!student_name || !class_roll || !class_name || !gender) {
      return res.status(500).send({ message: 'All fields are required' });
    }

    // Images File Handling
    // Images File Handling
    const imageUrls = {};
    const files = req.files;

    if (files) {
      for (const key in files) {
        const file = files[key][0];
        const { path } = file;
        const result = await cloudinary.uploader.upload(path);
        imageUrls[key] = {
          url: result.secure_url,
          public_id: result.public_id,
        };
        fs.unlinkSync(path);
      }
    }

    const student = await Student.create({
      student_name,
      student_name_bd,
      father_name,
      mother_name,
      father_name_nid,
      mother_name_nid,
      uniq_id,
      sub_scholar_id,
      scholar_id,
      class_name,
      class_roll,
      phone_number,
      address,
      date_of_birth,
      birth_reg_no,
      gender,
      blood_group,
      image: imageUrls.image,
      birth_image: imageUrls.birth_image,
    });

    res.status(201).send({
      success: true,
      message: 'Student created successfully',
      student,
    });
  } catch (error) {
    res.status(404).send({
      success: false,
      message: 'Something wrong!! Try again',
    });
    next(error);
    console.error(error);
  }
};

//get all Student
export const getStudent = async (req, res) => {
  try {
    const students = await Student.find({}).sort('class_roll');

    res.status(200).send({
      success: true,
      message: 'All notices get successfully ',
      students,
    });
  } catch (error) {
    console.log(error);
  }
};

//delete Student
export const deleteStudent = async (req, res) => {
  try {
    const id = req.params.id;
    const student = await Student.findById({ _id: id });

    if (!student) {
      return res.status(404).send({
        success: false,
        message: 'Student not found',
      });
    }
    // Delete images from Cloudinary
    if (student.image) {
      for (const image of student.image) {
        const { public_id } = image;
        await cloudinary.uploader.destroy(public_id);
      }
    }

    // Delete birth image from Cloudinary
    if (student.birth_image) {
      for (const birthImage of student.birth_image) {
        const { public_id } = birthImage;
        await cloudinary.uploader.destroy(public_id);
      }
    }

    await Student.findByIdAndDelete(id);

    res.status(200).send({
      success: true,
      message: 'Student deleted successfully',
    });
  } catch (error) {
    console.log(error);
    res.status(500).send({
      success: false,
      message: 'Error while deleting Student',
      error,
    });
  }
};

//update product
export const updateStudent = async (req, res) => {
  try {
    const {
      student_name,
      student_name_bd,
      father_name,
      mother_name,
      father_name_nid,
      mother_name_nid,
      uniq_id,
      sub_scholar_id,
      scholar_id,
      class_name,
      class_roll,
      phone_number,
      address,
      date_of_birth,
      birth_reg_no,
      gender,
      blood_group,
    } = req.body;

    const id = req.params.id;
    // Validation
    if (!student_name || !class_roll || !class_name || !gender) {
      return res.status(500).send({ message: 'All fields are required' });
    }

    const oldStudent = await Student.findById({ _id: id });

    // Image handling
    const files = req.files;
    const imageUrls = files?.image ? [] : oldStudent.image;
    const birthImageUrls = files?.birth_image ? [] : oldStudent.birth_image;

    if (files?.image) {
      // Delete old images
      for (const image of oldStudent.image) {
        const { public_id } = image;
        await cloudinary.uploader.destroy(public_id);
      }

      // Upload new images
      for (const file of files.image) {
        const { path } = file;
        const result = await cloudinary.uploader.upload(path);
        const data = {
          url: result.secure_url,
          public_id: result.public_id,
        };
        imageUrls.push(data);
        fs.unlinkSync(path);
      }
    }

    if (files?.birth_image) {
      // Delete old birth images
      for (const birthImage of oldStudent.birth_image) {
        const { public_id } = birthImage;
        await cloudinary.uploader.destroy(public_id);
      }

      // Upload new birth images
      for (const file of files.birth_image) {
        const { path } = file;
        const result = await cloudinary.uploader.upload(path);
        const data = {
          url: result.secure_url,
          public_id: result.public_id,
        };
        birthImageUrls.push(data);
        fs.unlinkSync(path);
      }
    }

    const student = await Student.findByIdAndUpdate(
      id,
      {
        student_name,
        student_name_bd,
        father_name,
        mother_name,
        father_name_nid,
        mother_name_nid,
        uniq_id,
        sub_scholar_id,
        scholar_id,
        class_name,
        class_roll,
        phone_number,
        address,
        date_of_birth,
        birth_reg_no,
        gender,
        blood_group,
        image: imageUrls,
        birth_image: birthImageUrls,
      },
      { new: true }
    );

    await student.save();

    res.status(201).send({
      success: true,
      message: 'Student Updated Successfully',
      student,
    });
  } catch (error) {
    res.status(404).send({
      success: false,
      message: 'Something wrong!! Try again',
      error: error,
    });
    console.error(error);
  }
};
